#include "MTSimpleResource.h"
#include "MTExceptions.h"
#include "MTFileList.h"
#include "MTApplication.h"
#include "MTNumberEditor.h"
#include "MTStringEditor.h"

#include <stack>

MTSimpleResourceHandler::MTSimpleResourceHandler()
{
	localDataCopy = nil;
	localDataSize = 0;
}

MTSimpleResourceHandler::~MTSimpleResourceHandler()
{
	if(localDataCopy)
	{
		DisposePtr((Ptr)localDataCopy);
		localDataCopy = nil;
	}
}

void MTSimpleResourceHandler::ReparseData(void)
{
	MTSimpleResourceEntry			* instructionsTraverse;
	UInt8							* dataTraverse;
	SInt32							bytesLeft;
	UInt32							entryData;
	UInt32							variables[0x40];
	UInt32							dataArea1, dataArea2;
	std::stack <LoopStackRecord>	loopStack;
	MTSimpleResourceInfoWindow		* infoWindow = dynamic_cast<MTSimpleResourceInfoWindow *>(this);
	UInt8							dividerStatus = 0;
	
	resourceEntries.clear();
	
	instructionsTraverse = GetDataTemplate()->entries;
	dataTraverse = localDataCopy;
	bytesLeft = localDataSize;
	
	while(instructionsTraverse->data && (bytesLeft > 0))
	{
		entryData = instructionsTraverse->data;
		dataArea1 = (entryData >> kTypeDataArea1Shift) & kTypeDataArea1ShiftMask;
		dataArea2 = (entryData >> kTypeDataArea2Shift) & kTypeDataArea2ShiftMask;
		
		if(entryData & kTypeStartSection)
		{
			UInt32	loopTimes = variables[dataArea1];
			
			if(loopTimes == 0)
			{
				UInt32	levels = 0;
				
				do
				{
					if(instructionsTraverse->data & kTypeStartSection)
						levels++;
					if(instructionsTraverse->data & kTypeEndSection)
						levels--;
					
					instructionsTraverse++;
				}
				while((levels > 0) && instructionsTraverse->data);
			}
			else
			{
				LoopStackRecord	stackRecord;
				
				stackRecord.loopsRemaining = variables[dataArea1] - 1;
				stackRecord.loopStart = instructionsTraverse + 1;
				
				loopStack.push(stackRecord);
			}
			
			dividerStatus = 1;
		}
		
		if(entryData & kTypeEndSection)
		{
			if(loopStack.empty())
				throw MTException("MTSimpleResourceHandler::ParseData: loop stack underrun");
			
			if(loopStack.top().loopsRemaining > 0)
			{
				loopStack.top().loopsRemaining--;
				instructionsTraverse = loopStack.top().loopStart;
				
				dividerStatus = 1;
				
				continue;
			}
			else
			{
				loopStack.pop();
			}
		}
		
		if(entryData & kTypeSpecialType)
		{
			switch(dataArea1)
			{
				case kCustomTypeCString:
				{
					MTParsedResourceEntry	entry;
					
					entry.format =	entryData;
					entry.data =	(UInt32)dataTraverse;
					entry.info =	instructionsTraverse->info;
					entry.title =	instructionsTraverse->title;
					entry.extend =	instructionsTraverse->extend;
					entry.basePtr =	(void *)dataTraverse;
					entry.divider =	dividerStatus;
					dividerStatus = 0;
					
					resourceEntries.push_back(entry);
					
					dataTraverse += dataArea2;
					bytesLeft -= dataArea2;
				}
				break;
				
				case kCustomTypeFloat:
				{
					MTParsedResourceEntry	entry;
					
					entry.format =	entryData;
					entry.data =	ResourceReadData(dataTraverse, 4, 1, 1);
					entry.info =	instructionsTraverse->info;
					entry.extend =	instructionsTraverse->extend;
					entry.title =	instructionsTraverse->title;
					entry.basePtr =	(void *)dataTraverse;
					entry.divider =	dividerStatus;
					dividerStatus = 0;
					
					resourceEntries.push_back(entry);
					
					dataTraverse += 4;
					bytesLeft -= 4;
				}
				break;
				
				case kCustomTypeJumpTo:
				{
					UInt32	newOffset, currentOffset;
					SInt32	extend;
					
					extend =		instructionsTraverse->extend;
					currentOffset =	dataTraverse - localDataCopy;
					
					switch(dataArea2)
					{
						case kCustomTypeJumpToAbsolute:
							newOffset = extend;
							break;
						
						case kCustomTypeJumpToRelative:
							newOffset = currentOffset + extend;
							break;
						
						case kCustomTypeJumpToVariable:
							if(	(extend < 0x00) ||
								(extend > 0x30))
								throw MTException("MTSimpleResourceHandler::ParseData: Invalid variable index");
							
							newOffset = variables[extend];
							break;
					}
					
					if(newOffset >= localDataSize)
						throw MTException("MTSimpleResourceHandler::ParseData: Jump to invalid offset");
					
					dataTraverse = localDataCopy + newOffset;
					bytesLeft = localDataSize - newOffset;
				}
				break;
				
				case kCustomTypeHexDump:
				{
					UInt32	dumpSize = instructionsTraverse->extend;
					
					if(dumpSize > bytesLeft)
						dumpSize = bytesLeft;
					
					MTParsedResourceEntry	entry;
					
					entry.format =	entryData;
					entry.data =	dumpSize;
					entry.info =	instructionsTraverse->info;
					entry.extend =	instructionsTraverse->extend;
					entry.title =	instructionsTraverse->title;
					entry.basePtr =	(void *)dataTraverse;
					entry.divider =	dividerStatus;
					dividerStatus = 0;
					
					resourceEntries.push_back(entry);
					
					dataTraverse += dumpSize;
					bytesLeft -= dumpSize;
				}
				break;
				
				case kCustomTypeRegisterOp:
				{
					UInt32	uExtend = instructionsTraverse->extend;
					UInt32	lhs = (uExtend >>  0) & 0x3F;
					UInt32	rhs = (uExtend >>  6) & 0x3F;
					UInt32	tgt = (uExtend >> 12) & 0x3F;
					UInt32	tgtData, lhsData, rhsData;
					
					lhsData = variables[lhs];
					rhsData = variables[rhs];
					
					switch(dataArea2)
					{
						case kCustomTypeROpAdd:			tgtData = lhsData + rhsData;	break;
						case kCustomTypeROpSubtract:	tgtData = lhsData - rhsData;	break;
						case kCustomTypeROpMultiply:	tgtData = lhsData * rhsData;	break;
						case kCustomTypeROpDivide:		tgtData = lhsData / rhsData;	break;
						case kCustomTypeROpSkip:		dataTraverse += lhsData;
														bytesLeft -= lhsData;			break;
						default: throw MTException("MTSimpleResourceHandler::ParseData: unknown rop");
					}
					
					variables[tgt] = tgtData;
				}
				break;
			}
		}
		
		if(entryData & kTypeBitSizeMask)
		{
			UInt32	numBytes = 1 << ((entryData & kTypeBitSizeMask) - 1);
			UInt32	data;
			
			data = ResourceReadData(dataTraverse, numBytes, entryData & kTypeSwapBytes, entryData & kTypeIsUnsigned);
			
			if(entryData & kTypeIsFiller)
			{
				if(entryData & kTypeDataBlock)
				{
					numBytes *= dataArea1;
				}
			}
			else
			{
				MTParsedResourceEntry	entry;
				
				entry.format =	entryData;
				entry.data =	data;
				entry.info =	instructionsTraverse->info;
				entry.title =	instructionsTraverse->title;
				entry.extend =	instructionsTraverse->extend;
				entry.basePtr =	(void *)dataTraverse;
				entry.divider =	dividerStatus;
				dividerStatus = 0;
				
				resourceEntries.push_back(entry);
			}
			
			if(entryData & kTypeStoreToVariable)
			{
				SInt32	adjustment = dataArea2;
				
				if(adjustment & 0x00000300)
					adjustment |= ~0x000003FF;
				
				variables[dataArea1] = data + adjustment;
			}
			
			if((entryData & kTypeSpecialType) && (dataArea1 == kCustomTypeJumpIf))
			{
				UInt8	doJump = 0;
				
				if(dataArea2 & kCustomTypeJumpIfNoAdvance)
					numBytes = 0;
				
				if(dataArea2 & kCustomTypeJumpIfEqual)
					doJump |= (data == instructionsTraverse->extend);
				
				if(dataArea2 & kCustomTypeJumpIfLessThan)
					doJump |= (data < instructionsTraverse->extend);
				
				if(dataArea2 & kCustomTypeJumpIfGreaterThan)
					doJump |= (data > instructionsTraverse->extend);
				
				if(doJump)
				{
					UInt32	jumpTarget;
					
					if(dataArea2 & kCustomTypeJumpIfTerminate)
						goto done;
					
					jumpTarget = (instructionsTraverse->info >> kTypeInfoDataShift) & kTypeInfoDataMask;
					
					instructionsTraverse = GetDataTemplate()->entries + jumpTarget;
					instructionsTraverse--;
				}
			}
			
			dataTraverse += numBytes;
			bytesLeft -= numBytes;
		}
		
		if((entryData & kTypeSpecialType) && (dataArea1 == kCustomTypeJumpAbs))
		{
			UInt32	jumpTarget;
			
			jumpTarget = (instructionsTraverse->info >> kTypeInfoDataShift) & kTypeInfoDataMask;
			
			instructionsTraverse = GetDataTemplate()->entries + jumpTarget;
			instructionsTraverse--;
		}
		
		if(entryData & kTypeStoreConstantToVariable)
		{
			variables[dataArea1] = dataArea2;
		}
		
		instructionsTraverse++;
	}
	
	done:
	
	PostProcess();
	PrivatePostProcess();
	
	if(infoWindow)
	{
		infoWindow->HandleUpdateEvent();
	}
}

void MTSimpleResourceHandler::ParseData(UInt8 * basePtr, UInt32 size)
{
	localDataSize = size;
	localDataCopy = (UInt8 *)NewPtr(size);
	if(!localDataCopy)
		throw MTMemoryException("Out of memory");
	
	BlockMove(basePtr, localDataCopy, size);
	
	DisposePtr((Ptr)basePtr);
	
	ReparseData();
}

MTSimpleResourceInfoWindow::MTSimpleResourceInfoWindow()
{
	exportOffset = 0;
	exportSize = 0;
	exportFile = 0;
	
	isDirty = 0;
}

MTSimpleResourceInfoWindow::~MTSimpleResourceInfoWindow()
{
	if(isDirty)
		WriteOutData();
}

void MTSimpleResourceInfoWindow::PrivatePostProcess(void)
{
	char	buf[4096];
	
	SetNumEntries(resourceEntries.size());
	
	for(UInt32 i = 0; i < resourceEntries.size(); i++)
	{
		ResourcePrintData(buf, &resourceEntries[i]);
		
		SetEntryData(i, buf, 0);
		SetEntryTitle(i, resourceEntries[i].title.c_str(), 0);
		SetEntryDivider(i, resourceEntries[i].divider, 0);
		
		switch(resourceEntries[i].info & kTypeInfoTypeMask)
		{
			case kTypeInfoResourceOffset:
				exportOffset = resourceEntries[i].data;
				
				if(exportFile == 0)
					exportFile = (resourceEntries[i].info >> kTypeInfoFlagUsesShift) & kTypeInfoFlagUsesMask;
				break;
			
			case kTypeInfoResourceSize:
				exportSize = resourceEntries[i].data;
				
				if(exportFile == 0)
					exportFile = (resourceEntries[i].info >> kTypeInfoFlagUsesShift) & kTypeInfoFlagUsesMask;
				break;
		}
	}
	
	RecalculateListSpacing();
}

void MTSimpleResourceInfoWindow::GetWindowName(StringPtr theString)
{
	char	buf[4096];
	
	if(viewerResourceName.empty())
		std::sprintf(buf, "Info: %.4s #%d", &viewerResourceType, viewerResourceIdx);
	else
		std::sprintf(buf, "Info: %.4s %s", &viewerResourceType, viewerResourceName.c_str());
	
	CopyCStringToPascal(buf, theString);
}

void MTSimpleResourceInfoWindow::RecieveMessage(UInt32 messageType, UInt32 messageData)
{
	#pragma unused (messageData)
	
	switch(messageType)
	{
		case 'EXPT':
			if(	(selection < resourceEntries.size()) &&
				(resourceEntries[selection].info & kTypeInfoTypeID))
			{
				MTFileList	* theFileList;
				
				theFileList = GetOwningFileList();
				
				if(theFileList)
					theFileList->theFile->ExportFileByIdx(theFileList->theFile->LookupIdxFromID(resourceEntries[selection].data));
			}
			else
			{
				if(exportFile)
				{
					DoExport();
				}
			}
			break;
		
		case 'INFO':
		{
			if(	(selection < resourceEntries.size()) &&
				(resourceEntries[selection].info & kTypeInfoTypeID))
			{
				MTFileList	* theFileList;
				
				theFileList = GetOwningFileList();
				
				if(theFileList)
					theFileList->SpawnFileInfoWindow(theFileList->theFile->LookupIdxFromID(resourceEntries[selection].data));
			}
		}
		break;
		
		case 'EDIT':
			SpawnEditorWindow(selection);
			break;
		
		case 'IMPT':
			DoImport();
			break;
	}
}

UInt8 MTSimpleResourceInfoWindow::SupportsMessage(UInt32 messageType, UInt32 messageData)
{
	#pragma unused (messageData)
	
	switch(messageType)
	{
		case 'EXPT':
			return	(exportFile != 0) ||
					((selection < resourceEntries.size()) && (resourceEntries[selection].info & kTypeInfoTypeID));
		
		case 'INFO':
			return((selection < resourceEntries.size()) && (resourceEntries[selection].info & kTypeInfoTypeID));
		
		case 'EDIT':
			return 1;
	}
	
	return 0;
}

void MTSimpleResourceInfoWindow::HandleListDoubleClick(UInt32 selection)
{
	if(selection < resourceEntries.size())
	{
		MTParsedResourceEntry	* entry = &resourceEntries[selection];
		
		UInt32	type = entry->info & kTypeInfoTypeMask;
		
		switch(type)
		{
			case kTypeInfoTypeID:
			{
				MTFileList	* fileList = GetOwningFileList();
				
				if(fileList)
				{
					UInt32	idx;
					
					idx = fileList->theFile->LookupIdxFromID(entry->data);
					
					fileList->OpenViewerWindow(idx);
				}
			}
			break;
			
			case kTypeInfoResourceOffset:
			case kTypeInfoResourceSize:
			{
				if(entry->info & kTypeInfoFlagIsParsed)
				{
					MTFileList	* fileList = GetOwningFileList();
					
					if(fileList)
					{
						UInt32		type;
						std::string	spawnName;
						UInt8		base = 0;
						
						switch(entry->info & (kTypeInfoFlagUsesMask << kTypeInfoFlagUsesShift))
						{
							case kTypeInfoFlagUsesBaseFile:		base = kSourceFileBase;		break;
							case kTypeInfoFlagUsesSegmentFile:	base = kSourceFileSegment;	break;
							case kTypeInfoFlagUsesRawFile:		base = kSourceFileRaw;		break;
						}
						
						if((entry->extend) || (viewerResourceName.size() < 4))
						{
							type = entry->extend;
							spawnName = entry->title;
						}
						else
						{
							type =	(viewerResourceName[0] << 24) |
									(viewerResourceName[1] << 16) |
									(viewerResourceName[2] <<  8) |
									(viewerResourceName[3] <<  0);
							spawnName = entry->title.substr(4, viewerResourceName.size());
						}
						
						fileList->OpenViewerWindowEX(type, (char *)spawnName.c_str(), 0, base, exportOffset, exportSize);	// *hack*
					}
				}
			}
			break;
		}
	}
}

void MTSimpleResourceInfoWindow::DoExport(void)
{
	if(exportFile)
	{
		MTFileList	* fileList = GetOwningFileList();
		
		if(fileList)
		{
			MTFile	* theFile = fileList->theFile;
			
			switch(exportFile << kTypeInfoFlagUsesShift)
			{
				case kTypeInfoFlagUsesBaseFile:
					theFile->ExportFile(exportOffset, exportSize);
					break;
				
				case kTypeInfoFlagUsesSegmentFile:
					theFile->ExportSegment(exportOffset, exportSize);
					break;
				
				case kTypeInfoFlagUsesRawFile:
					theFile->ExportRaw(exportOffset, exportSize);
					break;
			}
		}
	}
}

void MTSimpleResourceInfoWindow::DoImport(void)
{
	MTFileList	* fileList = GetOwningFileList();
	
	if(fileList && fileList->theFile)
		fileList->theFile->ImportFileByIdx(viewerResourceIdx);
}

void MTSimpleResourceInfoWindow::WriteOutData(void)
{
	MTFileList	* theFileList = GetOwningFileList();
	IFileStream	* theFileStream = nil;
	
	switch(viewerResourceFile)
	{
		case kSourceFileBase:
			theFileStream = theFileList->theFile->OpenBaseFile();
			break;
		
		case kSourceFileRaw:
			theFileStream = theFileList->theFile->OpenRawFile();
			break;
		
		case kSourceFileSegment:
			theFileStream = theFileList->theFile->OpenDataFile();
			break;
	}
	
	if(theFileStream)
	{
		theFileStream->SetPosition(viewerResourceOffset);
		theFileStream->WriteBuffer(localDataCopy, viewerResourceSize);
		
		theFileStream->CloseFile();
		
		delete theFileStream;
	}
}

void MTSimpleResourceInfoWindow::SpawnEditorWindow(UInt32 idx)
{
	MTParsedResourceEntry	* theEntry = GetResourceEntry(idx);
	MTEditor				* theEditor = nil;
	
	if(theEntry->format & kTypeSpecialType)
	{
		switch((theEntry->format >> kTypeDataArea1Shift) & kTypeDataArea1ShiftMask)
		{
			case kCustomTypeCString:
				theEditor = new MTStringEditor;
				break;
			
			case kCustomTypeFloat:
				theEditor = new MTNumberEditor;
				break;
		}
	}
	else
	{
		theEditor = new MTNumberEditor;
	}
	
	if(theEditor)
	{
		theEditor->SetVariant(theEntry->format);
		theEditor->SetBasePtr(theEntry->basePtr);
		theEditor->SetTitle(&theEntry->title);
		
		theEditor->SetOwner(this);
		
		gTheApp->AttachWindow(theEditor);
		
		isDirty = 1;
	}
}

UInt32 ResourceReadData(UInt8 * basePtr, UInt32 format)
{
	return ResourceReadData(basePtr, 1 << ((format & kTypeBitSizeMask) - 1), format & kTypeSwapBytes, format & kTypeIsUnsigned);
}

UInt32 ResourceReadData(UInt8 * basePtr, UInt32 numBytes, UInt32 swap, UInt32 isUnsigned)
{
	switch(numBytes)
	{
		case 1:
		{
			if(isUnsigned)
				return *((UInt8 *)basePtr);
			else
				return *((SInt8 *)basePtr);
		}
		break;
		
		case 2:
		{
			UInt16 data;
			
			data = *((UInt16 *)basePtr);
			
			if(swap)
			{
				data =	((data >> 8) & 0x00FF) |
						((data << 8) & 0xFF00);
			}
			
			if(isUnsigned)
				return (UInt16)data;
			else
				return (SInt16)data;
		}
		break;
		
		case 4:
		{
			UInt32 data;
			
			data = *((UInt32 *)basePtr);
			
			if(swap)
			{
				data =	((data >> 24) & 0x000000FF) |
						((data >>  8) & 0x0000FF00) |
						((data <<  8) & 0x00FF0000) |
						((data << 24) & 0xFF000000);
			}
			
			if(isUnsigned)
				return (UInt32)data;
			else
				return (SInt32)data;
		}
		break;
		
		default:
			throw MTException("MTSimpleResourceHandler::ResourceReadData: bad numBytes value");
			break;
	}
	
	return 0;
}

void ResourceWriteData(UInt8 * basePtr, UInt32 format, UInt32 data)
{
	ResourceWriteData(basePtr, 1 << ((format & kTypeBitSizeMask) - 1), format & kTypeSwapBytes, format & kTypeIsUnsigned, data);
}

void ResourceWriteData(UInt8 * basePtr, UInt32 numBytes, UInt32 swap, UInt32 isUnsigned, UInt32 data)
{
	switch(numBytes)
	{
		case 1:
		{
			data &= 0x000000FF;
			
			if(isUnsigned)
				*((UInt8 *)basePtr) = data;
			else
				*((SInt8 *)basePtr) = data;
		}
		break;
		
		case 2:
		{
			data &= 0x0000FFFF;
			
			if(swap)
			{
				data =	((data >> 8) & 0x00FF) |
						((data << 8) & 0xFF00);
			}
			
			if(isUnsigned)
				*((UInt16 *)basePtr) = data;
			else
				*((SInt16 *)basePtr) = data;
		}
		break;
		
		case 4:
		{
			if(swap)
			{
				data =	((data >> 24) & 0x000000FF) |
						((data >>  8) & 0x0000FF00) |
						((data <<  8) & 0x00FF0000) |
						((data << 24) & 0xFF000000);
			}
			
			if(isUnsigned)
				*((UInt32 *)basePtr) = data;
			else
				*((SInt32 *)basePtr) = data;
		}
		break;
		
		default:
			throw MTException("MTSimpleResourceHandler::ResourceWriteData: bad numBytes value");
	}
}

void ResourcePrintData(char * buf, MTParsedResourceEntry * entry)
{
	char	formatString[256];
	UInt32	dataArea1;
	UInt32	dataArea2;
	
	dataArea1 = (entry->format >> kTypeDataArea1Shift) & kTypeDataArea1ShiftMask;
	dataArea2 = (entry->format >> kTypeDataArea2Shift) & kTypeDataArea2ShiftMask;
	
	if(entry->format & kTypeSpecialType)
	{
		switch(dataArea1)
		{
			case kCustomTypeCString:
				std::sprintf(formatString, "%s%d%c", "%.", dataArea2, 's');
				break;
			
			case kCustomTypeFloat:
				float	* temp1 = (float *)&entry->data;	// dirty hack
				double	temp2 = *temp1;
				
				#if DEBUG_LOG
				std::sprintf(buf, "%f (%.8X)", temp2, entry->data);
				#else
				std::sprintf(buf, "%f", temp2);
				#endif
				return;
			
			case kCustomTypeHexDump:
				std::strcpy(buf, "<hex data>");
				return;
		}
	}
	else
	{
		formatString[0] = '%';
		
		if(entry->format & kTypeDisplayAsDec)
		{
			if(entry->format & kTypeIsUnsigned)
			{
				formatString[1] = 'l';
				formatString[2] = 'u';
				formatString[3] = 0;
			}
			else
			{
				formatString[1] = 'l';
				formatString[2] = 'd';
				formatString[3] = 0;
			}
		}
		else
		{
			formatString[1] = '.';
			formatString[2] = ((1 << ((entry->format & kTypeBitSizeMask) - 1)) * 2) + '0';
			formatString[3] = 'X';
			formatString[4] = 0;
		}
	}
	
	if(entry->info & kTypeInfoTypeID_String)
	{
		WindowPtr	theFrontWindow = FrontWindow();
		MTWindow	* theFrontMTWindow = GetMTWindowFromWindowPtr(theFrontWindow);
		
		if(theFrontMTWindow)
		{
			MTFileList	* frontFileList = theFrontMTWindow->GetOwningFileList();
			
			if(frontFileList)
			{
				MTFile	* theFile = frontFileList->theFile;
				UInt8	* baseStringData;
				
				baseStringData = theFile->LoadFileByID(entry->data);
				
				if(baseStringData)
				{
					char	* stringData;
					char	formatString2[256];
					
					stringData = (char *)&baseStringData[8];
					stringData[0x80 - 1] = 0;
					
					std::sprintf(formatString2, "%s %%s", formatString);
					std::sprintf(buf, formatString2, entry->data, stringData);
					
					DisposePtr((Ptr)baseStringData);
					
					return;
				}
			}
		}
	}
	
	std::sprintf(buf, formatString, entry->data);
}

void ResourcePrintDataSimple(char * buf, UInt32 data, UInt32 format)
{
	char	formatString[256];
	UInt32	dataArea1;
	UInt32	dataArea2;
	
	dataArea1 = (format >> kTypeDataArea1Shift) & kTypeDataArea1ShiftMask;
	dataArea2 = (format >> kTypeDataArea2Shift) & kTypeDataArea2ShiftMask;
	
	if(format & kTypeSpecialType)
	{
		switch(dataArea1)
		{
			case kCustomTypeCString:
				std::sprintf(formatString, "%s%d%c", "%.", dataArea2, 's');
				break;
			
			case kCustomTypeFloat:
				float	* temp1 = (float *)&data;	// dirty hack
				double	temp2 = *temp1;
				
				#if DEBUG_LOG
				std::sprintf(buf, "%f (%.8X)", temp2, data);
				#else
				std::sprintf(buf, "%f", temp2);
				#endif
				return;
			
			case kCustomTypeHexDump:
				std::strcpy(buf, "<hex data>");
				return;
		}
	}
	else
	{
		formatString[0] = '%';
		
		if(format & kTypeDisplayAsDec)
		{
			if(format & kTypeIsUnsigned)
			{
				formatString[1] = 'l';
				formatString[2] = 'u';
				formatString[3] = 0;
			}
			else
			{
				formatString[1] = 'l';
				formatString[2] = 'd';
				formatString[3] = 0;
			}
		}
		else
		{
			formatString[1] = '.';
			formatString[2] = ((1 << ((format & kTypeBitSizeMask) - 1)) * 2) + '0';
			formatString[3] = 'X';
			formatString[4] = 0;
		}
	}
	
	std::sprintf(buf, formatString, data);
}
